"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeOptions = normalizeOptions;
exports.coerceTypes = coerceTypes;
exports.createTargets = createTargets;
exports.build = build;
exports.configureBuildCommand = configureBuildCommand;

function _builderUtil() {
  const data = require("builder-util");

  _builderUtil = function () {
    return data;
  };

  return data;
}

function _chalk() {
  const data = _interopRequireDefault(require("chalk"));

  _chalk = function () {
    return data;
  };

  return data;
}

function _electronBuilderLib() {
  const data = require("electron-builder-lib");

  _electronBuilderLib = function () {
    return data;
  };

  return data;
}

function _bluebirdLst() {
  const data = _interopRequireDefault(require("bluebird-lst"));

  _bluebirdLst = function () {
    return data;
  };

  return data;
}

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/** @internal */
function normalizeOptions(args) {
  if (args.extraMetadata != null) {
    throw new (_builderUtil().InvalidConfigurationError)("Please specify extraMetadata under config field");
  }

  if (args.targets != null) {
    return args;
  }

  if (args.draft != null || args.prerelease != null) {
    _builderUtil().log.warn({
      solution: "set releaseType (http://electron.build/configuration/publish#GithubOptions-releaseType) in the GitHub publish options"
    }, "--draft and --prerelease is deprecated");
  }

  let targets = new Map();

  function processTargets(platform, types) {
    function commonArch(currentIfNotSpecified) {
      if (platform === _electronBuilderLib().Platform.MAC) {
        return args.x64 || currentIfNotSpecified ? [_builderUtil().Arch.x64] : [];
      }

      const result = Array();

      if (args.x64) {
        result.push(_builderUtil().Arch.x64);
      }

      if (args.armv7l) {
        result.push(_builderUtil().Arch.armv7l);
      }

      if (args.arm64) {
        result.push(_builderUtil().Arch.arm64);
      }

      if (args.ia32) {
        result.push(_builderUtil().Arch.ia32);
      }

      return result.length === 0 && currentIfNotSpecified ? [(0, _builderUtil().archFromString)(process.arch)] : result;
    }

    if (args.platform != null) {
      throw new (_builderUtil().InvalidConfigurationError)(`--platform cannot be used if --${platform.buildConfigurationKey} is passed`);
    }

    if (args.arch != null) {
      throw new (_builderUtil().InvalidConfigurationError)(`--arch cannot be used if --${platform.buildConfigurationKey} is passed`);
    }

    let archToType = targets.get(platform);

    if (archToType == null) {
      archToType = new Map();
      targets.set(platform, archToType);
    }

    if (types.length === 0) {
      const defaultTargetValue = args.dir ? [_electronBuilderLib().DIR_TARGET] : [];

      for (const arch of commonArch(args.dir === true)) {
        archToType.set(arch, defaultTargetValue);
      }

      return;
    }

    for (const type of types) {
      const suffixPos = type.lastIndexOf(":");

      if (suffixPos > 0) {
        (0, _builderUtil().addValue)(archToType, (0, _builderUtil().archFromString)(type.substring(suffixPos + 1)), type.substring(0, suffixPos));
      } else {
        for (const arch of commonArch(true)) {
          (0, _builderUtil().addValue)(archToType, arch, type);
        }
      }
    }
  }

  if (args.mac != null) {
    processTargets(_electronBuilderLib().Platform.MAC, args.mac);
  }

  if (args.linux != null) {
    processTargets(_electronBuilderLib().Platform.LINUX, args.linux);
  }

  if (args.win != null) {
    processTargets(_electronBuilderLib().Platform.WINDOWS, args.win);
  }

  if (targets.size === 0) {
    if (args.platform == null && args.arch == null) {
      processTargets(_electronBuilderLib().Platform.current(), []);
    } else {
      targets = createTargets(normalizePlatforms(args.platform), args.dir ? _electronBuilderLib().DIR_TARGET : null, args.arch);
    }
  }

  const result = Object.assign({}, args);
  result.targets = targets;
  delete result.dir;
  delete result.mac;
  delete result.linux;
  delete result.win;
  delete result.platform;
  delete result.arch;
  const r = result;
  delete r.m;
  delete r.o;
  delete r.l;
  delete r.w;
  delete r.windows;
  delete r.macos;
  delete r.$0;
  delete r._;
  delete r.version;
  delete r.help;
  delete r.c;
  delete result.ia32;
  delete result.x64;
  delete result.armv7l;
  delete result.arm64;
  let config = result.config; // config is array when combining dot-notation values with a config file value (#2016)

  if (Array.isArray(config)) {
    const newConfig = {};

    for (const configItem of config) {
      if (typeof configItem === "object") {
        (0, _builderUtil().deepAssign)(newConfig, configItem);
      } else if (typeof configItem === "string") {
        newConfig.extends = configItem;
      }
    }

    config = newConfig;
    result.config = newConfig;
  }

  if (config != null && typeof config !== "string") {
    if (config.extraMetadata != null) {
      coerceTypes(config.extraMetadata);
    }

    if (config.mac != null) {
      // ability to disable code sign using -c.mac.identity=null
      coerceValue(config.mac, "identity");
    }
  }

  if ("project" in r && !("projectDir" in result)) {
    result.projectDir = r.project;
    delete r.project;
  }

  return result;
}

function coerceValue(host, key) {
  const value = host[key];

  if (value === "true") {
    host[key] = true;
  } else if (value === "false") {
    host[key] = false;
  } else if (value === "null") {
    host[key] = null;
  } else if (key === "version" && typeof value === "number") {
    host[key] = value.toString();
  } else if (value != null && typeof value === "object") {
    coerceTypes(value);
  }
}
/** @private */


function coerceTypes(host) {
  for (const key of Object.getOwnPropertyNames(host)) {
    coerceValue(host, key);
  }

  return host;
}

function createTargets(platforms, type, arch) {
  const targets = new Map();

  for (const platform of platforms) {
    const archs = platform === _electronBuilderLib().Platform.MAC ? [_builderUtil().Arch.x64] : arch === "all" ? [_builderUtil().Arch.x64, _builderUtil().Arch.ia32] : [(0, _builderUtil().archFromString)(arch == null ? process.arch : arch)];
    const archToType = new Map();
    targets.set(platform, archToType);

    for (const arch of archs) {
      archToType.set(arch, type == null ? [] : [type]);
    }
  }

  return targets;
}

function build(rawOptions) {
  const buildOptions = normalizeOptions(rawOptions || {});
  const packager = new (_electronBuilderLib().Packager)(buildOptions);
  let electronDownloader = null;

  packager.electronDownloader = options => {
    if (electronDownloader == null) {
      electronDownloader = _bluebirdLst().default.promisify(require("electron-download-tf"));
    }

    return electronDownloader(options);
  };

  return (0, _electronBuilderLib().build)(buildOptions, packager);
}
/**
 * @private
 * @internal
 */


function configureBuildCommand(yargs) {
  const publishGroup = "Publishing:";
  const buildGroup = "Building:";
  const deprecated = "Deprecated:";
  return yargs.option("mac", {
    group: buildGroup,
    alias: ["m", "o", "macos"],
    description: `Build for macOS, accepts target list (see ${_chalk().default.underline("https://goo.gl/5uHuzj")}).`,
    type: "array"
  }).option("linux", {
    group: buildGroup,
    alias: "l",
    description: `Build for Linux, accepts target list (see ${_chalk().default.underline("https://goo.gl/4vwQad")})`,
    type: "array"
  }).option("win", {
    group: buildGroup,
    alias: ["w", "windows"],
    description: `Build for Windows, accepts target list (see ${_chalk().default.underline("https://goo.gl/jYsTEJ")})`,
    type: "array"
  }).option("x64", {
    group: buildGroup,
    description: "Build for x64",
    type: "boolean"
  }).option("ia32", {
    group: buildGroup,
    description: "Build for ia32",
    type: "boolean"
  }).option("armv7l", {
    group: buildGroup,
    description: "Build for armv7l",
    type: "boolean"
  }).option("arm64", {
    group: buildGroup,
    description: "Build for arm64",
    type: "boolean"
  }).option("dir", {
    group: buildGroup,
    description: "Build unpacked dir. Useful to test.",
    type: "boolean"
  }).option("publish", {
    group: publishGroup,
    alias: "p",
    description: `Publish artifacts (to GitHub Releases), see ${_chalk().default.underline("https://goo.gl/tSFycD")}`,
    choices: ["onTag", "onTagOrDraft", "always", "never", undefined]
  }).option("draft", {
    group: deprecated,
    description: "Please set releaseType in the GitHub publish options instead",
    type: "boolean",
    default: undefined
  }).option("prerelease", {
    group: deprecated,
    description: "Please set releaseType in the GitHub publish options instead",
    type: "boolean",
    default: undefined
  }).option("platform", {
    group: deprecated,
    description: "The target platform (preferred to use --mac, --win or --linux)",
    choices: ["mac", "win", "linux", "darwin", "win32", "all", undefined]
  }).option("arch", {
    group: deprecated,
    description: "The target arch (preferred to use --x64 or --ia32)",
    choices: (0, _builderUtil().getArchCliNames)().concat("all", undefined)
  }).option("prepackaged", {
    alias: ["pd"],
    group: buildGroup,
    description: "The path to prepackaged app (to pack in a distributable format)"
  }).option("projectDir", {
    alias: ["project"],
    group: buildGroup,
    description: "The path to project directory. Defaults to current working directory."
  }).option("config", {
    alias: ["c"],
    group: buildGroup,
    description: "The path to an electron-builder config. Defaults to `electron-builder.yml` (or `json`, or `json5`), see " + _chalk().default.underline("https://goo.gl/YFRJOM")
  }).group(["help", "version"], "Other:").example("electron-builder -mwl", "build for macOS, Windows and Linux").example("electron-builder --linux deb tar.xz", "build deb and tar.xz for Linux").example("electron-builder --win --ia32", "build for Windows ia32").example("electron-builder -c.extraMetadata.foo=bar", "set package.json property `foo` to `bar`").example("electron-builder --config.nsis.unicode=false", "configure unicode options for NSIS");
}

function normalizePlatforms(rawPlatforms) {
  const platforms = rawPlatforms == null || Array.isArray(rawPlatforms) ? rawPlatforms : [rawPlatforms];

  if (platforms == null || platforms.length === 0) {
    return [_electronBuilderLib().Platform.fromString(process.platform)];
  } else if (platforms[0] === "all") {
    if (process.platform === _electronBuilderLib().Platform.MAC.nodeName) {
      return [_electronBuilderLib().Platform.MAC, _electronBuilderLib().Platform.LINUX, _electronBuilderLib().Platform.WINDOWS];
    } else if (process.platform === _electronBuilderLib().Platform.LINUX.nodeName) {
      // macOS code sign works only on macOS
      return [_electronBuilderLib().Platform.LINUX, _electronBuilderLib().Platform.WINDOWS];
    } else {
      return [_electronBuilderLib().Platform.WINDOWS];
    }
  } else {
    return platforms.map(it => it instanceof _electronBuilderLib().Platform ? it : _electronBuilderLib().Platform.fromString(it));
  }
} 
//# sourceMappingURL=builder.js.map